/*
 *	smreg.cpp
 *
 *	Smaky Registry / INTERFACE.
 *
 *	Smaky Registry: Implementation of the publicly available
 *	interface, which can be built into a Smaky library.
 *
 *	(C) Copyright 1996-1997, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#include "smreg.h"
#include "registry.h"
#include "regkey.h"
#include "reghive.h"
#include "regdata.h"

/*****************************************************************************/

static Registry* g_registry   = 0;
static RegHive*  g_user_hive  = 0;
static Card32    g_open_count = 0;
static Card32    g_lock[2]    = { 0, 0 };

/*****************************************************************************/

extern int g_dummy_regddhive;

/*****************************************************************************/

Reg::Reg ()
{
	g_dummy_regddhive++;	//	required for the linker to bind `regddhive.o'
}

Reg::~Reg ()
{
}

Reg*
Reg::GetRegistry ()
{
	static Reg reg;
	return & reg;
}

/*****************************************************************************/

/*
 *	Find the proper key pointer from the key ID. This will work only
 *	if the key has been entered.
 */

static RegKey*
make_key_from_entered_id (Card32 key_id)
{
	return (key_id) ? g_registry->FindKeyFromID (key_id) : 0;
}

/*****************************************************************************/

Card32
Reg::Open ()
{
	AutoLock autolock (g_lock);
	USE (autolock);

	if (g_registry == 0) {
		g_registry   = new Registry;
		g_user_hive  = new RegHive (g_registry);
	}

	g_open_count++;
	
	RegKey* key = g_registry->Root ();
	key->Enter ();
	key->SetHive (g_user_hive);
	
	return g_registry->KeyID (key);
}

void
Reg::Close ()
{
	Card32 the_lock[2] = { 0, 0 };
	
	{
		AutoLock autolock (g_lock);
		USE (autolock);
		
		RegKey* g_key = g_registry->Root ();
		g_key->Release ();
		
		g_open_count--;

		if (g_open_count == 0) {
			
			g_registry->SetRoot (0);
			
			delete g_key;
			delete g_user_hive;
			delete g_registry;
			
			g_registry  = 0;
			g_user_hive = 0;
			the_lock    = g_lock;
			g_lock[0]   = 0;
			g_lock[1]   = 0;
		}
	}

	if (the_lock[0]) {
		AutoLock::KillLock (the_lock);
	}
}

/*****************************************************************************/

Card32
Reg::CreateKey (Card32 key_id, const char* name, Card32 hive_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegHive* hive    = (RegHive*)(hive_id);
	RegKey*  key     = make_key_from_entered_id (key_id);
	void*    dispose = 0;
	RegValue value;
	
	if (key == 0) {
		return 0;
	}
	
	if (hive == 0) {
		hive = key->GetHive ();
	}
	
	value.type        = REG_NAM_STRING;
	value.name.string = name;
	
	if ( (key->Find (value, 0, dispose))
	  && ((value.type & 0x000000FF) == REG_TYP_X_LEVEL)
	  && (value.data.level.key)
	  && (value.data.level.key->Enter ()) ) {
		
		Card32 id = g_registry->KeyID (value.data.level.key);
		if (dispose) key->Dispose (dispose);
		return id;
	}
	
	if (dispose) key->Dispose (dispose);
	
	RegKey* add = new DataRegKey (g_registry, key);
	
	if (add == 0) {
		return 0;
	}
	
	if (add->Enter ()) {
		
		value.type            = REG_TYP_X_LEVEL | REG_NAM_STRING;
		value.name.string     = name;
		value.data.level.key  = add;
		value.data.level.temp = 0;
		
		key->Insert (value, hive);
		
	} else {
		
		delete add;
		add = 0;
	}
	
	return g_registry->KeyID (add);
}

Card32
Reg::EnterKey (Card32 key_id, const char* name, Card32* hive_ptr)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegKey*  key     = make_key_from_entered_id (key_id);
	RegHive* hptr    = 0;
	void*    dispose = 0;
	RegValue value;
	
	if (key == 0) {
		return 0;
	}
	
	value.type        = REG_NAM_STRING;
	value.name.string = name;
	
	if (key->Find (value, &hptr, dispose)) {
		
		while ((value.type & 0x000000FF) == REG_TYP_X_DDATA) {
			value.data.dynamic.data->FillRegValue (value);
		}
		
		if ( ((value.type & 0x000000FF) == REG_TYP_X_LEVEL)
		  && (value.data.level.key)
		  && (value.data.level.key->Enter ()) ) {
			if (hive_ptr) hive_ptr[0] = (Card32)(hptr);
			Card32 id = g_registry->KeyID (value.data.level.key);
			if (dispose) key->Dispose (dispose);
			return id;
		}
	}
	
	if (dispose) key->Dispose (dispose);
	
	return 0;
}

Bool
Reg::ReleaseKey (Card32 key_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegKey* key = make_key_from_entered_id (key_id);
	
	if (key == 0) {
		return 0;
	}
	
	return key->Release ();
}

Bool
Reg::KillKeyOrValue (Card32 key_id, const char* name)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegValue value;
	RegKey*  key = make_key_from_entered_id (key_id);
	
	if (key == 0) {
		return 0;
	}
	
	value.type        = REG_NAM_STRING;
	value.name.string = name;
	
	return key->Remove (value);
}

Bool
Reg::Explore (Card32 key_id, Card32& index, RegValue& value, Card32* hive_ptr)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	Card32 buf_len  = 0;
	char*  buf_ptr  = 0;
	Card32 need_len = 0;
	
	if (value.type & REG_TYP_ARRAY) {
		buf_len = RegHive::GetDataBytes (&value);
		buf_ptr = (char*)(value.data.array.data);
	}
	
	RegKey*  key     = make_key_from_entered_id (key_id);
	RegHive* hive    = 0;
	void*    dispose = 0;
	
	if (key == 0) {
		return FALSE;
	}
	
	if (key->Find (index, value, &hive, dispose)) {
		
		while ((value.type & 0x000000FF) == REG_TYP_X_DDATA) {
			value.data.dynamic.data->FillRegValue (value);
		}
		
		if ((value.type & 0x000000FF) == REG_TYP_X_LEVEL) {
			value.data.level.key  = 0;
			value.data.level.temp = 0;
		}
		
		if (value.type & REG_TYP_ARRAY) {
			
			//	Arrays are not transmitted by pointer, but by copy. This is done
			//	only if the caller provided enough memory for it. If not, just
			//	return a null pointer to the array.
			
			need_len = RegHive::GetDataBytes (&value);
			
			if ( (buf_ptr)
			  && (buf_len >= need_len) ) {
				memcpy (buf_ptr, value.data.array.data, need_len);
			}
			
			value.data.array.data = buf_ptr;
		}
		
		if (hive_ptr) hive_ptr[0] = (Card32)(hive);
		if (dispose) key->Dispose (dispose);
		
		return TRUE;
	}
	
	if (dispose) key->Dispose (dispose);
	return FALSE;
}

Bool
Reg::CreateValue (Card32 key_id, RegValue& value, Card32 hive_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegKey*  key  = make_key_from_entered_id (key_id);
	RegHive* hive = (RegHive*)(hive_id);
	
	if (key == 0) {
		return FALSE;
	}
	
	if (hive == 0) {
		hive = key->GetHive ();
	}
	
	return key->Insert (value, hive);
}


Bool
Reg::CreateDynamicValue (Card32 key_id, const char* name,
						 Card32 c_id, const char* data, Card32 len,
						 Card32 hive_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegKey*  key  = make_key_from_entered_id (key_id);
	RegHive* hive = (RegHive*)(hive_id);
	
	if (key == 0) {
		return FALSE;
	}
	
	if (hive == 0) {
		hive = key->GetHive ();
	}
	
	// TODO: allocate the dynamic object according to the "c_id" class ID.
	
	return FALSE;
}


/*
 *	Find a value (not a key) in the specified level of the registry.
 *	This might build temporary data, which will be copied to the
 *	user's buffer and then discarded back again.
 */

Bool
Reg::GetValue (Card32 key_id, RegValue& value, Card32* hive_ptr)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	Card32 buf_len  = 0;
	char*  buf_ptr  = 0;
	Card32 need_len = 0;
	
	if (value.type & REG_TYP_ARRAY) {
		buf_len = RegHive::GetDataBytes (&value);
		buf_ptr = (char*)(value.data.array.data);
	}
	
	RegKey*  key     = make_key_from_entered_id (key_id);
	RegHive* hive    = 0;
	void*    dispose = 0;
	
	if (key == 0) {
		return FALSE;
	}
	
	if (key->Find (value, &hive, dispose)) {
		
		while ((value.type & 0x000000FF) == REG_TYP_X_DDATA) {
			value.data.dynamic.data->FillRegValue (value);
		}
		
		if ((value.type & 0x000000FF) == REG_TYP_X_LEVEL) {
			value.data.level.key  = 0;
			value.data.level.temp = 0;
		}
		
		if (value.type & REG_TYP_ARRAY) {
			
			//	Arrays are not transmitted by pointer, but by copy. This is done
			//	only if the caller provided enough memory for it. If not, just
			//	return a null pointer to the array.
			
			need_len = RegHive::GetDataBytes (&value);
			
			if ( (buf_ptr)
			  && (buf_len >= need_len) ) {
				memcpy (buf_ptr, value.data.array.data, need_len);
			}
			
			value.data.array.data = buf_ptr;
		}
		
		if (hive_ptr) hive_ptr[0] = (Card32)(hive);
		if (dispose) key->Dispose (dispose);
		
		return TRUE;
	}
	
	if (dispose) key->Dispose (dispose);
	return FALSE;
}

Card32
Reg::GetArrayBytes (const RegValue& value)
{
	return RegHive::GetDataBytes (&value);
}

const char*
Reg::GetName (const RegValue& value)
{
	Card32      type = (value.type) & 0xFF000000;
	const char* text = "?";
	
	switch (type) {
		case REG_NAM_TOKEN:
			break;
		
		case REG_NAM_ATOM:
		case REG_NAM_STRING:
			text = value.name.string;
			break;
	}

	return text;
}

/*****************************************************************************/

Card32
Reg::LoadHive (Card32 hive_id, Card32 key_id, void* data)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegKey*  key  = make_key_from_entered_id (key_id);
	RegHive* hive = (RegHive*)(hive_id);
	
	if (key == 0) {
		return 0;
	}
	
	if (hive == 0) {
		hive = new RegHive (g_registry);
		if (hive == 0) {
			return 0;
		}
	}
	
	if (hive->ImportData (data, key)) {
		if (hive_id == 0) {
			delete hive;
		}
		return 0;
	}

	return (Card32)(hive);
}

Card32
Reg::GetGlobalHive ()
{
	return (Card32)(g_user_hive);
}

Bool
Reg::SaveHive (Card32 hive_id, int (*func)(const void*, Card32, void*), void* arg)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	if (hive_id == 0) {
		hive_id = (Card32)(g_user_hive);
	}
	
	RegKey*     root   = g_registry->Root ();
	RegHive*    hive   = (RegHive*)(hive_id);
	HiveStream* stream = 0;
	
	hive->SaveBegin (stream);
	stream->SetWriter (func, arg);
	
	while (stream && stream->Saving ()) {
		root->SaveHiveTree (hive, stream);
	}

	return TRUE;
}

Card32
Reg::CreateHive ()
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	RegHive* hive = new RegHive (g_registry);
	return (Card32)(hive);
}

Bool
Reg::KillHive (Card32 hive_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
	if (hive_id == 0) {
		return FALSE;
	}
	
	RegKey*  root = g_registry->Root ();
	RegHive* hive = (RegHive*)(hive_id);

	if (hive == g_user_hive) {
		return FALSE;
	}
	
	root->KillHiveTree (hive);
	delete hive;

	return TRUE;
}

Bool
Reg::HasHiveChanged (Card32 hive_id)
{
	AutoLock autolock (g_lock);
	USE (autolock);
	
//	RegKey*  root = g_registry->Root ();
	RegHive* hive = (RegHive*)(hive_id);
	
	if (hive == 0) hive = g_user_hive;
	
	return hive->HasChanged ();
}

