/*
 *	registry.h
 *
 *	Smaky Registry: definitions for publicly available entry points,
 *	data types and constants.
 *
 *	(C) Copyright 1996, Pierre ARNAUD, OPaC bright ideas
 *		CH-1437 Suscevaz
 */

#ifndef _REGISTRY_H_
#define	_REGISTRY_H_

#include <C++/portable.h>
#include <stddef.h>

/*****************************************************************************/

/*
 *	Forward definitions; internal classes, etc.
 */

class RegKey;
struct RegValue;

/*****************************************************************************/

/*
 *	Name and type definitions for the SmRegValues::type field.
 *	You may OR these together (name/type/array)
 */

#define	REG_NAM_TOKEN	 0x01000000	//	name is inlined by a token
#define	REG_NAM_ATOM	 0x02000000	//	name is specified by an atom
#define	REG_NAM_STRING	 0x03000000	//	name is specified by string
#define REG_NAM_RESERVED 0xFF000000	//	name field is reserved

#define	REG_TYP_ARRAY	 0x00000100	//	an array of X (num/ptr)

#define	REG_TYP_BOOLEAN	 0x00000001	//	just 0 or 1
#define	REG_TYP_BYTE	 0x00000002	//	an 8-bit byte (any kind of data)
#define	REG_TYP_LATIN	 0x00000003	//	an 8-bit Latin character
#define	REG_TYP_UNICODE	 0x00000004	//	a 16-bit Unicode character
#define	REG_TYP_INTEGER	 0x00000005	//	a 32-bit signed value
#define	REG_TYP_CARDINAL 0x00000006	//	a 32-bit unsigned value
#define	REG_TYP_REAL	 0x00000007	//	a 64-bit IEEE real value

#define	REG_TYP_X_LEVEL	 0x000000E0	//	internal: next level
#define	REG_TYP_X_LINK	 0x000000FD	//	internal: link to next level
#define	REG_TYP_X_STRING 0x000000FE	//	internal: string table
#define	REG_TYP_X_CROSS	 0x000000FF	//	internal: cross reference

/*****************************************************************************/

/*
 *	Internal data types: don't use these yourself.
 */

struct RegXcross
{
	RegValue*		next;			//	internal: next value
	RegValue*		prev;			//	internal: previous value
};

struct RegXlink
{
	RegValue*		value;			//	internal: start of next level
	Card32			temp;			//	internal: temporary storage
};

struct RegXlevel
{
	RegKey*			key;			//	internal: next key (one level down)
	Card32			temp;			//	internal: temporary storage
};

/*****************************************************************************/

/*
 *	Array of elements. This kind of data will be found whenever
 *	an array is stored in the registry. It is made of a pointer
 *	and a counter, which does not represent an amount of bytes,
 *	but the number of elements stored in the array (the size
 *	may thus be a fraction or a multiple of this number).
 */

struct RegArray
{
	void*			data;			//	pointer to data in array
	Card32			num;			//	number of elements in array
};


/*
 *	Data stored in the registry can be of several different
 *	types (boolean => 1-bit .. real => 64-bit).
 */

union RegAnyData
{
	Bool			boolean;		//	type is REG_TYP_BOOLEAN
	Int32			integer;		//	type is REG_TYP_INTEGER
	Card32			cardinal;		//	type is REG_TYP_CARDINAL
	Real			real;			//	type is REG_TYP_REAL
	RegArray		array;			//	type is REG_TYP_xxx | REG_TYP_ARRAY
	RegXcross		cross;			//	type is REG_TYP_X_CROSS
	RegXlink		link;			//	type is REG_TYP_X_LINK
	RegXlevel		level;			//	type is REG_TYP_X_LEVEL
};


/*
 *	Each value/key in the registry has an associated name. This
 *	name can be either a token, an atom (for each internal name
 *	it will be an atom) or a pointer to a string.
 */

union RegName
{
	Card32			token;			//	token of name
	const char*		atom;			//	atom of name (internal string)
	const char*		string;			//	null-terminated string
	Card32			count;			//	internal: counter
};


/*
 *	A value is the association of a type, a name and a data
 *	structure which stores the data itself or a pointer to it.
 */

struct RegValue
{
	Card32			type;			//	type description
	RegName			name;			//	name or pointer to name
	RegAnyData		data;			//	data (1/8/16/32/64-bit)
};

/*****************************************************************************/

#define	REG_MAGIC_TYPE	0xFFFFFFFF

#define	REG_ERR_OK		0x0000
#define	REG_ERR_CORRUPT	0x0001
#define	REG_ERR_FULL	0x0002
#define	REG_ERR_ILLOP	0x0003

/*****************************************************************************/

#define	X_TYPE(x)	((x) & 0x00FF)
#define	X_ATTR(x)	((x) & 0xFF00)
#define	X_NAME(x)	((x) & 0xFF000000)

/*****************************************************************************/

/*
 *	This class provides the methods needed to read/write data from/to a key
 *	of the registry. The data is exchanged through a compact representation
 *	as stored in the internal structures (type + name + data).
 */

class Reg
{

protected:
	Reg ();
	
public:
	virtual ~ Reg ();
	
	static Reg* GetRegistry ();
	
	virtual Card32 Open () = 0;
	virtual void Close () = 0;
	
	virtual Card32 CreateKey (Card32 key, const char* name, Card32 hive = 0) = 0;
	virtual Card32 EnterKey (Card32 key, const char* name, Card32* hive_ptr = 0) = 0;
	virtual Bool ReleaseKey (Card32 key) = 0;
	
	virtual Bool Explore (Card32 key, Card32& index, RegValue& value, Card32* hive_ptr = 0) = 0;
	
	virtual Bool CreateValue (Card32 key, RegValue& value, Card32 hive = 0) = 0;
	virtual Bool KillKeyOrValue (Card32 key, const char* name) = 0;
	virtual Bool GetValue (Card32 key, RegValue& value, Card32* hive_ptr = 0) = 0;
	
	virtual Card32 GetArrayBytes (const RegValue& value) = 0;
	virtual const char* GetName (const RegValue& value) = 0;
	
	virtual Card32 GetGlobalHive () = 0;
	
	virtual Card32 CreateHive () = 0;
	virtual Bool   KillHive (Card32 hive) = 0;
	virtual Bool   HasHiveChanged (Card32 hive) = 0;
	
	virtual Card32 LoadHive (Card32 hive, Card32 key, void* data) = 0;
	virtual Bool   SaveHive (Card32 hive, int (*)(const void*, Card32, void*), void*) = 0;
};

#endif /* _REGISTRY_H_ */

