/*
 *	makerom.cc
 *
 *	Source of MAKEROM, a utility which produces ROM files for the Smaky by
 *	merging binary files together.
 *
 *	(C) Copyright 1996-1997, Pierre Arnaud, OPaC bright ideas
 *		CH-1437 Suscvaz, Switzerland
 */


#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <c++/fos.h>


unsigned long _stksize   = 10*1024;
unsigned long _SMASOVER  = 1;
unsigned char _SMAPRIO   = 12;
unsigned char _SMAREV    = 0;
unsigned char _SMAVER    = 6;
unsigned long _SMAUNIT   = 0x000000A7;
unsigned short _SMADIS[] = { 20, 2000, 20, 2000 };
char _SMAIDE[] = "(C) 1996-1997 Pierre ARNAUD, OPaC bright ideas, CH-1437 Suscvaz";


#define	ROM_BIN_FILE		(0x0001)		//	name is for a binary file


Bool	g_verbose  = FALSE;
Bool	g_trap     = FALSE;
Bool	g_report   = FALSE;
Bool	g_exclam   = FALSE;
int		g_revision = 0;
int		g_version  = 0;
FILE*	g_rep_file = 0;
FILE*	g_dep_file = 0;

struct ROMFileDesc
{
	char		name[16];
	Card32		attr;
	SmakyDate	date;
};


/*
 *	Parse one line and extract information about the source name,
 *	the destination name and the associated flags.
 */

void
parse_line (const char*& source, char* src_name, char* dst_name, Card32& flags)
{
	char* s = src_name;
	char* d = dst_name;
	
	src_name[0] = 0;
	dst_name[0] = 0;
	flags       = 0;
	int  count  = 0;
	Bool space  = TRUE;
	
	while ( (*source)
	     && (*source != '\n')
	     && (*source != '\r') ) {
		
		if ( (*source == ' ')
		  || (*source == '\t') ) {
			
			if (space == FALSE) {
				count++;
			}
			
			if (count == 2) {
				while ( (*source)
				     && (*source != '\n')
				     && (*source != '\r') ) {
					source++;
				}
				break;
			}
			
			space = TRUE;
			source++;
			continue;
		}
		
		if (*source == ';') {
			
			if (g_rep_file) {
				fprintf (g_rep_file, " <TR VALIGN=top BGCOLOR=#FFFFC0><TD COLSPAN=5><H2>");
				source++;
				while (*source == ' ') source++;
			}
			
			while ( (*source)
			     && (*source != '\n')
			     && (*source != '\r') ) {
				
				char c = *source++;
				
				if (g_rep_file) {
					fprintf (g_rep_file, "%c", c);
				}
			}
			
			if (g_rep_file) {
				fprintf (g_rep_file, "</H2></TD></TR>\n");
			}
			
			break;
		}
		
		if (count == 0) {
			*src_name++ = *source++;
		} else {
			*dst_name++ = *source++;
		}
		
		space = FALSE;
	}
	
	source++;
	
	*src_name = 0;
	*dst_name = 0;
	
	if (d[0] == 0) {
		strcpy (d, s);
	}
	
	if ( (s[0])
	  && (s[0] != '!') ) {
		flags = ROM_BIN_FILE;
	}
}


/*
 *	Read a file without any kind of conversion. Add a <0> at the
 *	end of it. Return also the size, if requested.
 */

Bool
read_file (const char* name, char*& data, Card32& size, SmakyArgs* p_arg = 0)
{
	Card32    mode    = FOS_OPEN_READ | FOS_OPEN_ARGS;
	SmakyArgs args    = { 0 };
	Card16    channel = 0;
	Card16    error   = 0;
	char      other[200];
	
	error = FOS::ArgsOpen (name, mode, args, channel);
	
	if (error) {
		strcpy (other, "#:");
		strcat (other, name);
		error = FOS::ArgsOpen (other, mode, args, channel);
	}
	
	if (error) {
		errno = error;
		data  = 0;
		size  = 0;
		p_arg = 0;
		return FALSE;
	}
	

if (g_trap) { asm volatile ("trap #0"); }

	size = args.length;
	data = new char[size+1];

	error = FOS::Read (channel, size, data);
	
	FOS::Close (channel);
	
	if (error) {
		delete data;
		errno = error;
		data  = 0;
		size  = 0;
		p_arg = 0;
		return FALSE;
	}
	
	data[size] = 0;
	
	if (p_arg) {
		*p_arg = args;
	}
	
	return TRUE;
}

Bool
read_file (const char* name, char*& data)
{
	Card32 size = 0;
	return read_file (name, data, size);
}

Bool
read_src_file (const char* name, char*& data)
{
	if (read_file (name, data)) {
		if (g_exclam) {
			if (g_dep_file) { fclose (g_dep_file); g_dep_file = 0; }
			char dep_name[200];
			strcpy (dep_name, name);
			char* p = dep_name;
			Bool ok = FALSE;
			while (*p) { if (*p == '.') { *p = '!'; ok = TRUE; } p++; }
			if (ok) g_dep_file = fopen (dep_name, "wt");
			if (g_dep_file) {
				const char* p  = 0;
				const char* p1 = name;
				const char* p2 = dep_name;
				
				p = p1; while (*p) { if (*p == ':') p1 = p+1; p++; }
				p = p2; while (*p) { if (*p == ':') p2 = p+1; p++; }
				
				fprintf (g_dep_file, "%s/D\n%s/D\n\n", p1, p2);
			}
		}
		return TRUE;
	}
	return FALSE;
}

Bool
read_file (const char* name, char*& data, SmakyArgs& args)
{
	Card32 size = 0;
	return read_file (name, data, size, &args);
}


/*
 *	Parse the script file and generate an EPROM file.
 */

Bool
parse_file (const char* script, Card16 channel)
{
	SmakyArgs args = { 0 };
	char      src_name[200];
	char      dst_name[200];
	Card16    error = 0;
	char*     data  = 0;
	Card32    size  = 0;
	
	while (*script) {
		
		Card32 flags = 0;
		parse_line (script, src_name, dst_name, flags);
		
		if (flags & ROM_BIN_FILE) {
			
			if (g_verbose) {
				printf ("- ", dst_name);
			}
			
			if (strlen (dst_name) > 15) {
				fprintf (stderr, "Name %s too long.\n", dst_name);
				errno = 0;
				return FALSE;
			}
			
			if (read_file (src_name, data, args) == FALSE) {
				fprintf (stderr, "Could not read %s\n", src_name);
				return FALSE;
			}
			
			if (g_verbose) {
				printf ("Inserting file %s.\n", dst_name);
			}
			
			if (g_dep_file) {
				fprintf (g_dep_file, "%s\n", src_name);
			}
			
			ROMFileDesc desc = { { 0 }, args.attr, args.d_create };
			Card32      len  = sizeof (desc);
			Card32      size = (args.length + 1) & 0xFFFFFFFE;
			
			strcpy (desc.name, dst_name);
			
			const char* ident  = (const char*)(data) + 128;
			const Card32* plen = (const Card32*)(data) + 3;
			
			if ( (data[0] == 0x02)
			  && (strcmp (ident, "MAKEROM-no-head") == 0) ) {
				
				//	Just write the file without its header. This is used for the
				//	very special boot file which comes at the very beginning of
				//	the EPROM image.
				
				FOS::Write (channel, size-256, data+256);
				
			} else {
				
				Card32 fos_size = (plen[0] + 256 + 1) & 0xFFFFFFFE;
				
				if ( (data[0] == 0x02)
				  && (size != fos_size) ) {
					printf ("Warning: %s is %06x bytes long, not %06x\n", dst_name, size, fos_size);
					size = fos_size;
				}
				
				char date[20];
				char time[20];
				
				sprintf (date, "%02x/%02x/%02x", args.d_create.day,
						 args.d_create.month, args.d_create.year);
				
				sprintf (time, "%02x:%02x:%02x", args.d_create.hour,
						 args.d_create.minute, args.d_create.second);
				
				if (g_rep_file) {
					fprintf (g_rep_file, " <TR VALIGN=top BGCOLOR=#FFFFFF>\n"
										 "  <TD ROWSPAN=2 ALIGN=left><CODE>%s</CODE></TD>\n"
										 "  <TD ALIGN=center>%d.%d</TD>\n"
										 "  <TD ALIGN=center>%s</TD>\n"
										 "  <TD ALIGN=center>%s</TD>\n"
										 "  <TD ALIGN=center>%s</TD>\n"
										 " </TR>\n"
										 " <TR VALIGN=top BGCOLOR=#FFFFFF>\n"
										 "  <TD ALIGN=left COLSPAN=4>%s&nbsp;</TD>\n"
										 " </TR>\n",
										 dst_name, data[0x18], data[0x19],
										 (data[0] & 0x04) ? "short" : "normal",
										 date, time,
										 (data[0] & 0x04) ? "not available" : ident);
				}
				
				//	Write the ROM header and the complete file.
				
				FOS::Write (channel, len, &desc);
				FOS::Write (channel, size, data);
			}
			
			delete data;
		}
	}
	
	ROMFileDesc desc = { { 0 }, 0, 0 };
	size = sizeof (desc);
	
	FOS::Write (channel, size, &desc);
	
	return TRUE;
}

void
fix_check (char* binary, Card32 size)
{
	Card32* obxlg  = (Card32*)(binary+0x0C);
	Card16* obxchk = (Card16*)(binary+0x02);
	Card8*  obxrev = (Card8*) (binary+0x18);
	Card8*  obxver = (Card8*) (binary+0x19);
	Card16  check  = 0;
	char*   data   = binary + 256;
	
	obxlg[0]  = size;
	obxrev[0] = g_revision;
	obxver[0] = g_version;

	asm volatile ( "movel %1,a2\n\t"
				   "movel %2,a4\n\t"
				   ".long 0x4E450067\n\t"		//	FOS ?CALCHECK
				   "movew d5,%0"
				 : "=g" (check)
				 : "g" (binary), "g" (data)
				 : "a2", "a4", "d5", "d7" );
	
	obxchk[0] = check;
}

int
main (int argc, char* argv[])
{
	int         num      = 0;
	const char* rom_name = "%m_rom.bin";
	const char* rep_name = "%m_rom.html";
	
	for (num = 1; num < argc; num++) {
		if (argv[num][0] == '-') {
			
			//	Option "-rom file" produces the `file' as output (target) file.
			
			if (strcmp (argv[num], "-rom") == 0) {
				rom_name    = argv[num+1];
				argv[num]   = 0;
				argv[num+1] = 0;
				num++;
				continue;
			}
			
			//	Option "-verbose" says what we are doing.
			
			if (strcmp (argv[num], "-verbose") == 0) {
				g_verbose = TRUE;
				argv[num] = 0;
				continue;
			}
			
			if (strcmp (argv[num], "-TRAP") == 0) {
				g_trap    = TRUE;
				argv[num] = 0;
				asm volatile ("trap #0");
				continue;
			}
			
			if (strcmp (argv[num], "-rep") == 0) {
				g_report  = TRUE;
				argv[num] = 0;
				continue;
			}
			
			if (strcmp (argv[num], "-dep") == 0) {
				g_exclam  = TRUE;
				argv[num] = 0;
				continue;
			}

			if (strcmp (argv[num], "-report") == 0) {
				g_report    = TRUE;
				rep_name    = argv[num+1];
				argv[num]   = 0;
				argv[num+1] = 0;
				num++;
				continue;
			}
			
			if ( (strncmp (argv[num], "-rev=", 4) == 0)
			  && (strchr (argv[num], '.') ) ) {
				g_revision = atoi (argv[num]+5);
				g_version  = atoi (strchr (argv[num]+5, '.')+1);
				argv[num] = 0;
				continue;
			}
			
			if (strcmp (argv[num], "-help") == 0) {
				printf ( "MakeRom %d.%d\n\n"
				         " -rom file         produces `file' as a ROM image\n"
				         " -rev=x.y          sets revision to x.y\n"
				         " -verbose          says what we are doing\n"
				         " -report file      generates an HTML report in `file'\n"
				         " -dep              generates `!' file for input GROM\n"
				         " -help             displays this help\n"
				         "\n"
				         "Default output file is %M_ROM.BIN\n",
				         _SMAREV, _SMAVER );
				return 0;
			}
		}
	}
	
	Card16    channel = 0;
	Card16    error   = 0;
	SmakyArgs args;
	Card32    mode    = FOS_OPEN_WRITE | FOS_OPEN_EXCL | FOS_OPEN_CREATE;
	char*     binhead = 0;
	Card32    size    = 256;
	
	if (read_file ("(#:,)makerom.bin", binhead) == FALSE) {
		fprintf (stderr, "Could not read MAKEROM.BIN !\n");
		return errno;
	}
	
	if (g_verbose) {
		printf ("Creating ROM image %s %d.%d\n", rom_name, g_revision, g_version);
	}
	
	error = FOS::ArgsOpen (rom_name, mode, args, channel);
	
	if (error) {
		return error;
	}
	
	FOS::Write (channel, size, binhead);
	
	if (g_report) {
		g_rep_file = fopen (rep_name, "wt");
		if (g_rep_file) {
			fprintf (g_rep_file, "<HTML><HEAD><TITLE>\n"
								 "Listing of %s revision %d.%d\n"
								 "</TITLE></HEAD>\n"
								 "<BODY>\n",
								 rom_name, g_revision, g_version);
			fprintf (g_rep_file, "<TABLE ALIGN=LEFT VALIGN=TOP WIDTH=100%% CELLSPACING=2>\n");
			fprintf (g_rep_file, " <TR VALIGN=top BGCOLOR=#000099>\n"
								 "  <TH ALIGN=center><FONT FACE=Arial COLOR=#ffffff>Name</FONT></TH>\n"
								 "  <TH ALIGN=center><FONT FACE=Arial COLOR=#ffffff>Rev.</FONT></TH>\n"
								 "  <TH ALIGN=center><FONT FACE=Arial COLOR=#ffffff>Header</FONT></TH>\n"
								 "  <TH ALIGN=center><FONT FACE=Arial COLOR=#ffffff>Date</FONT></TH>\n"
								 "  <TH ALIGN=center><FONT FACE=Arial COLOR=#ffffff>Time</FONT></TH>\n"
								 " </TR>\n");
		}
	}
	
	for (num = 1; num < argc; num++) {
		if (argv[num]) {
			char* data = 0;
			if ( (read_src_file (argv[num], data) == FALSE)
			  || (parse_file (data, channel) == FALSE) ) {
				FOS::Close (channel);
				FOS::Delete (rom_name);
				if (g_dep_file) { fclose (g_dep_file); g_dep_file = 0; }
				return error;
			}
			delete data;
			if (g_dep_file) { fclose (g_dep_file); g_dep_file = 0; }
		}
	}
	
	if (g_rep_file) {
		fprintf (g_rep_file, "</TABLE></BODY></HTML>\n");
		fclose (g_rep_file);
		g_rep_file = 0;
	}
	
	FOS::Trunc (channel);
	FOS::Close (channel);
	
	char* binary = 0;
	
	if (read_file (rom_name, binary, size) == FALSE) {
		fprintf (stderr, "Could not read back %s.\n", rom_name);
		return errno;
	}
	
	error = FOS::ArgsOpen (rom_name, mode, args, channel);
	
	if (error) {
		fprintf (stderr, "Could not write back %s.\n", rom_name);
		return error;
	}
	
	fix_check (binary, size-256);
	size = 256;
	
	FOS::Write (channel, size, binary);
	FOS::Close (channel);
	
	Card32 attr_x = 0x27FF | (1 << 0x000B);
	
	asm volatile ( "movel %0,d3\n\t"
				   "movel %1,a3\n\t"
				   ".long 0x4E45000A"					//	FOS ?CHATR
				 :
				 : "g" (attr_x), "g" (rom_name)
				 : "d3", "a3", "d7" );
	
	delete binary;
	
	if (g_verbose) {
		printf ("Successfully created %s\n", rom_name);
	}
	
	return 0;
}


